%% Procesamiento de Señales Biomédicas
%  Universidad de Granada
%  Grado en Ingeniería Electrónica Industrial
%
%  Tema 1. Fundamentos de Procesado Digital de Señales
%  Profesor: Joaquín T. Valderrama (jvalderrama@ugr.es)

%% Sección 1.d. Señales versus vectores
clear,clc                       % Reseteamos MATLAB
clf(figure(1)),hold on,box on   % Abrimos la figura 1
g = [4.5,0.5];                  % Vector g
x = [1,4];                      % Vector x
c = dot(g,x)/dot(x,x);          % Factor de escala que minimiza el error
ProyGsobreX = c*x;              % Proyección de g sobre x
e = g-ProyGsobreX;              % Vector error
plot([0,g(1)],[0,g(2)],'b')     % Vector g en azul
plot([0,x(1)],[0,x(2)],'m')     % Vector x en magenta
plot([0,ProyGsobreX(1)],[0,ProyGsobreX(2)],'g') % Proyección en verde
plot([g(1),ProyGsobreX(1)],[g(2),ProyGsobreX(2)],'r')   % Vector e en rojo
axis([0 5 0 5])                 % Fijamos el eje de la representación
legend('g','x','ProyGsobreX','e')               % Leyenda
fprintf(sprintf('El producto escalar entre x y e es: %1.6g\n',dot(x,e)))

%% Sección 1.e. Correlación de señales
% Paso 1. Lectura de archivos
% ECG source: https://www.physionet.org/content/ecg-arrhythmia/1.0.0/
clear,clc                       % Reseteamos MATLAB
clf(figure(1)),hold on,box on   % Abrimos la figura 1
File = load('ECG_Example.mat'); % Abrimos el archivo con ECG y fs
ECG = File.ECG;                 % Guardamos el ECG en una variable
fs = File.fs;                   % Guardamos la fs en una variable
ECG_Template = File.ECG_Template;   % Guardamos un beat de ECG limpio

% Paso 2. Representación de señales
subplot(3,1,1),box on
t_Template = 0:1/fs:(length(ECG_Template)-1)/fs;    % Eje de tiempos
plot(t_Template,ECG_Template,'r')       % Representamos el template
xlim([0 1])                             % Ventana de 1 segundo de duración
xlabel('Tiempo (s)')                    % Etiqueta del eje X
ylabel('Amplitud (\muV)')               % Etiqueta del eje Y
title('Plantilla de ECG de referencia') % título del subplot
subplot(3,1,2),box on
t = 0:1/fs:(length(ECG)-1)/fs;          % Eje de tiempos
sp2 = plot(t,ECG,'b',...                % Representamos el ECG en azul
    t_Template,ECG_Template+1500,'r');  % Y el template en rojo
xlabel('Tiempo (s)')                    % Etiqueta del eje X
ylabel('Amplitude (\muV)')              % Etiqueta del eje Y
title('Ejemplo de ECG')                 % Título de la figura
subplot(3,1,3),box on,hold on
ECG_Corr = nan(size(t));                % Inicialización de la correlación
sp3 = plot(t,ECG_Corr,'m');             % Plot de la correlación
xlabel('\tau (s)')
ylabel('< EGC , Plantilla Desplazada >')
title('Correlación de ECG con Plantilla desplazada \tau segundos')
axis([0 5 -1e7 2e7])
fprintf('Pulsa ENTER para realizar la correlación\n')
pause()

% Paso 3. Estimación de la correlación
for k=1:length(t)-length(ECG_Template)
    % Desplazamiento del beat de EEG
    ECG_Template_Desplazado = zeros(size(t));
    ECG_Template_Desplazado(k:k+length(ECG_Template)-1) = ECG_Template;
    
    % Producto escalar entre el ECG y la plantilla desplazada
    ECG_Corr(k) = dot(ECG_Template_Desplazado,ECG);

    % Representación
    set(sp2(1),'XData',t,'YData',ECG,'Color','b')
    set(sp2(2),'XData',t,'YData',ECG_Template_Desplazado+1500,'Color','r')
    set(sp3,'YData',ECG_Corr)
    pause(0.005)
end
fprintf('Pulsa ENTER para estimar máximos locales\n')
pause()

% Paso 4. Estimamos los máximos locales y trazamos su histograma
[PKS,LOCS] = findpeaks(ECG_Corr,fs);    % Estimación de máximos locales
plot(LOCS,PKS,'*k')                     % Visualización de máximos locales
clf(figure(2)),box on
histogram(PKS,10)                       % Histograma de amplitudes
xlabel('< EGC , Plantilla Desplazada >')
ylabel('Número de repeticiones')
title('Histograma de amplitudes de máximos locales')
Umbral = 8e6;                           % Fijamos un umbral razonable
Seleccion = PKS>Umbral;                 % Seleccionamos los picos > Umbral
ECG_Positions = LOCS(Seleccion);        % IDENTIFICACIÓN DE LATIDOS
fprintf('Pulsa ENTER para comparar el ECG original con uno modelado\n')
pause

% Paso 5. Pintamos nuestra estimación de latidos (ECG modelado)
ECG_Model = zeros(size(t));
for k=1:length(ECG_Positions)
    Position = round(ECG_Positions(k)*fs);
    ECG_Model(Position:Position+299) = ECG_Template;
end
clf(figure(3)),box on,hold on,grid on
plot(t,ECG,'b',t,ECG_Model,'r')
legend('ECG original','ECG modelado')
xlabel('Tiempo (s)')
ylabel('Amplitud (\muV)')
title('ECG original vs. modelado')


%% Sección 1.f. Convolución vs. correlación
clear, clc

% Definimos las funciones f y g
fs = 100;
t = -2:1/fs:2;
f = zeros(size(t)); cond = t>=-0.5 & t<=0.5;  f(cond) = 1;
g = zeros(size(t)); cond = t>=0 & t<=1;       g(cond) = exp(-t(cond)/2);

% Calculamos la correlación y la convolución
t2 = -4:1/fs:4;
Conv1 = conv(f,g);
Conv2 = conv(g,f);
Corr1 = xcorr(f,g);
Corr2 = xcorr(g,f);

% Representamos el resultado
clf(figure(1))
subplot(4,2,[1,2]),    plot(t,f),      xlim([-5 5]),   title('f(n)')
text(1,0.7,'\Pi(n)','FontSize',10)
subplot(4,2,[3,4]),    plot(t,g),      xlim([-5 5]),   title('g(n)')
xlabel('n','FontSize',10)
text(1.3,0.7,'e^{-n/2}','FontSize',10)
subplot(4,2,5),   plot(t2,Conv1),   xlim([-5 5])
title('Conv [f , g]'),              set(gca,'xtick',-5:5)
subplot(4,2,6),   plot(t2,Conv2),   xlim([-5 5])
title('Conv [g , f]'),              set(gca,'xtick',-5:5)
subplot(4,2,7),   plot(t2,Corr1),   xlim([-5 5])
title('Corr [f , g]'),              set(gca,'xtick',-5:5)
xlabel('n','FontSize',10), set(gca,'xtick',-5:5)
subplot(4,2,8),   plot(t2,Corr2),   xlim([-5 5])
title('Corr [g , f]'),              set(gca,'xtick',-5:5)
xlabel('n','FontSize',10)

%% Sección 2.a. TF inversa de la función triángulo
fs = 100;
B = 4;
t = -10:1/fs:10;
g = B/2 * sinc(B*t/2).^2;
plot(t,g)

%% Sección 2.e. Transformada Rápida de Fourier (FFT)
clear,clc
a = 2;                              % Índice de la exponencial
fs = 64;                            % Frecuencia de muestreo
T0 = 4;                             % Duración de g(t)
N0 = T0*fs;                         % Número de muestras
t = 0:1/fs:T0-1/fs;                 % Eje de tiempos
g = exp(-a*t);                      % Función g(t)
G = fft(g)/fs;                      % Función G(f) - Números complejos
                                    % Importante multiplicar por Ts (1/fs)
Gm_num = abs(G);                    % Magnitud de G(f), |G(f)|
Gp_num = angle(G);                  % Fase de G(f)

f = 0:fs/N0:fs-1/N0;                % Eje de frecuencias
G_analit = 1./(a+1i*2*pi*f);        % G(f) analítico
Gm_analit = abs(G_analit);          % Magnitud de G(f) analítica
Gp_analit = angle(G_analit);        % Fase de G(f) analítica

% Figuras
clf(figure(1)), plot(t,g,'.')           % Función g(t) en el TIEMPO
xlabel('Tiempo (s)'), ylabel('Amplitud'), title('g(t)')

clf(figure(2))                      % Función G(f) en la FRECUENCIA
subplot(211),hold on,box on         % Magnitud de G(f)
stem(f,Gm_num)                      % Gm numérica (FFT)
plot(f,Gm_analit)                   % Gm analítica
title('Magnitud de G(f)'), ylabel('Amplitud')
legend('Numérica (FFT)','Analítica')
xlim([0 8])
subplot(212),hold on,box on         % Fase de G(f)
stem(f,Gp_num)                      % Gp numérica (FFT)
plot(f,Gp_analit)                   % Gp analítica
set(gca,'ytick',[-pi/2,0,pi/2],'yticklabel',{'-\pi/2','0','\pi/2'})
axis([0 8 -2 2])
title('Fase de G(f)'), xlabel('Frecuencia (Hz)'), ylabel('radianes')


%% Sección 3.b. Comprobar la ortogonalidad de dos funciones sinc
clear,clc,

% Inicialización de variables
fs = 100;                       % fs para simular una señal analógica
t = -5:1/fs:5;                  % Eje de tiempo
Ts = 0.5;                       % Periodo de ceros en la función sinc
B = 1/(2*Ts);                   % 2·B·Ts = 1
k1 = 0;                         % Desplazamiento de f1
k2 = 1;                         % Desplazamiento de f2
A1 = 1;                         % Amplitud de f1
A2 = 1;                         % Amplitud de f2
f1 = A1*sinc(2*B*(t-k1*Ts));    % sinc(x) en MATLAB es sin(pi*x)/(pi*x)
f2 = A2*sinc(2*B*(t-k2*Ts));    % por lo que no hay que incluir pi.
ProdEsc = dot(f1,f2);           % Producto escalar

% Representación de señales
clf(figure(1))
plot(t,f1,t,f2),grid on
xlabel('Tiempo (s)')
ylabel('Amplitud')
title(sprintf('El producto escalar es %1.4f',ProdEsc))


%% Sección 3.d. Ruido de cuantificación de manera práctica y teórica
clear,clc

% Parámetros de la cuantificación
Nbits = 3;                          % Número de bits
L = 2^Nbits;                        % Número de niveles
mp = 1;                             % Rango de los valores de entrada
DeltaV = 2*mp/L;                    % Amplitud del subintervalo
Rangos = -mp:DeltaV:mp;             % Rangos
SampleLevels = (Rangos(2:end)+Rangos(1:end-1))/2;

% Señal original (alta resolución)
fs = 100;                           % Frecuencia de muestreo
t = 0:1/fs:1-1/fs;                  % Eje de tiempo
x = randn(size(t));                 % Señal ejemplo: Ruido blanco
x = x/max(abs(x))*mp;               % Normalizamos al rango de entrada

% Señal cuantificada (a la resolución de L niveles)
x_q = nan(size(x));
for k=1:length(x)
    Diff_Levels = abs(x(k)-SampleLevels);
    [~,Level] = min(Diff_Levels);
    x_q(k) = SampleLevels(Level);
end

% Estimación de la potencia del ruido de cuantificación
N0_Teo_dBm  = 10*log10(mp^2/(3*L^2)/0.001);     % Potencia teórica (dBm)
x_n = x-x_q;                                    % Ruido de cuantificación
N0_Prac = dot(x_n,x_n)/length(x_n);             % Potencia práctica 
N0_Prac_dBm = 10*log10(N0_Prac/0.001);          % Potencia práctica (dBm)

% Representación del resultado
clf(figure(1))
subplot(211),hold on,grid on,box on
plot(t,x,t,x_q)
set(gca,'ytick',[-mp,SampleLevels,mp])
xlabel('Tiempo (s)')
ylabel('Amplitud (V)')
legend('Original',sprintf('Cuantificada (%d bits)',Nbits))

subplot(212),hold on,grid on,box on
plot(t,x_n)
ylim([-mp,mp])
xlabel('Tiempo (x)')
ylabel('Amplitud (V)')
legend('Ruido de cuantificación')
title(sprintf('N0 teórico = %1.2f dBm    |    N0 práctico = %1.2f dBm',...
    N0_Teo_dBm,N0_Prac_dBm))


%% Diagrama de polos y zeros y su relación con la función de transferencia
clear,clc
Re = linspace(-2,2,200);        % Eje real
Im = linspace(-2,2,200);        % Eje imaginario
[X,Y] = meshgrid(Re,Im);        % Plano real (X) y plano imaginario (Y)
z = X + 1j*Y;                   % Plano complejo

% Filtro ejemplo diapositiva para x[n] = a^n u[n], con X(z) = 1/(1-0.7z^-1)
Gain = 1;
b = [1 0 0];
a = [1 -1.1 0];

% Filtro paso bajo Butterworth orden 1 con fs=48 kHz y fc=5600 Hz
% Gain = 0.277385657345731895251361720511340536177;       % Ganancia
% b = [1 1 0];                                            % Numerador
% a = [1 -0.445228685308536153986125327719491906464 0];   % Denominador

% Filtro paso alto Butterworth orden 2 con fs=48 kHz y fc=5600 Hz
% Gain = 0.591639895694974771167551352846203371882;
% b = [1 -2 1];
% a = [1 -1.008921623964990743615999235771596431732  ...
%     0.357637958814908007987298788066254928708];

% Función H(z) dados los coeficientes b y a
P = Gain * (b(1) + b(2)*z.^-1 + b(3)*z.^-2);
Q = a(1) + a(2)*z.^-1 + a(3)*z.^-2;
H = P./Q;

% Representación
clf(figure(1))
surf(X,Y,20*log10(abs(H)))      % Módulo
% surf(X,Y,angle(H))            % Fase
grid on, hold on
zlim([-100 100])
zlabel('|X(z)|')
xlabel('Real(z)')
ylabel('Imag(z)')
title('X(z) en el plano complejo z')

% Ejes
plot3([-2 2],[0 0],[0 0],'-r','lineWidth',2)    % Eje x
plot3([0 0],[-2 2],[0 0],'-r','lineWidth',2)    % Eje y

% Circunferencia unidad
Eje_Circ = linspace(-1,1,100);
plot3(Eje_Circ,+abs(sqrt(1-Eje_Circ.^2)),0*ones(size(Eje_Circ)),...
    '-r','lineWidth',2)
plot3(Eje_Circ,-abs(sqrt(1-Eje_Circ.^2)),0*ones(size(Eje_Circ)),...
    '-r','lineWidth',2)







