%% Procesamiento de Señales Biomédicas
%  Universidad de Granada
%  Grado en Ingeniería Electrónica Industrial
%
%  Tema 3. Electrocardiografía (ECG)
%  Profesor: Joaquín T. Valderrama (jvalderrama@ugr.es)

%% Sección 3.2.2.1 Filtrado FIR
clear,clc
FS = 12;

% Obtenemos un ECG de referencia
File = load('ECG_Example.mat');
ECG = File.ECG/1000;                % Señal ECG en mV
fs = File.fs;                       % Frecuencia de muestreo
clear File
N = length(ECG);                    % Número de muestras del ECG

% Simulamos una fuerte deriva de la línea de base 
t_w = (0:1/fs:(N-1)/fs)';
Deriva = 2.5*cos(2*pi*0.5*t_w);
ECG_Noise = ECG + Deriva;

% Diseño de filtros FIR con varios coeficientes
FIR_0050 = fir1(0050,2*0.6/fs,'high');
FIR_0500 = fir1(0500,2*0.6/fs,'high');
FIR_1000 = fir1(1000,2*0.6/fs,'high');
FIR_3000 = fir1(3000,2*0.6/fs,'high');

% Obtenemos la respuesta en frecuencia del filtro
h = FIR_3000;                           % Selección del filtro
f = 0:fs/N:fs-1/N;                      % Eje de frecuencias (Hz)
t_h = (0:1/fs:(length(h)-1)/fs)*1000;   % Eje de tiempos (ms)
H = freqz(h,1,f,fs);                    % Respuesta en frecuencia

% Comparamos las respuestas en frecuencia
clf(figure(1))
subplot(211)
plot(f,20*log10(abs(H))),grid on
xlim([0,3])
xlabel('Frecuencia (Hz)','FontSize',FS)
ylabel('Magnitud (dB)','FontSize',FS)
title('Filtro FIR de orden 3000','FontSize',FS)
subplot(212)
plot(t_h,h),grid on
xlabel('Tiempo (ms)','FontSize',FS)
ylabel('Amplitud','FontSize',FS)
title('Respuesta al impulso','FontSize',FS)

% Filtrado con filtro FIR de orden 3000 con compensación de retardo
Delay_Filter = ceil(length(FIR_3000)/2);
ECG_filtered = filter(FIR_3000,1,[ECG_Noise;zeros(4*fs,1)]);
ECG_filtered = ECG_filtered(Delay_Filter:Delay_Filter+length(ECG)-1);

% Comparación de la señal ECG sintetizada con la filtrada
clf(figure(2))
plot(t_w,ECG_Noise,t_w,ECG_filtered),grid on
lgnd = legend('Original','Filtrada');
set(lgnd,'FontSize',FS-2,'Location','SouthEast')
set(gca,'xtick',0:5,'FontSize',FS-2)
xlabel('Tiempo (s)','FontSize',FS)
ylabel('Amplitud (mV)','FontSize',FS)
title('Señal ECG con deriva de base sintetizada','FontSize',FS)


%% Sección 3.2.2.2 Filtrado IIR directo e inverso
clear,clc
FS = 12;

% Obtenemos un ECG de referencia
File = load('ECG_Example.mat');
ECG = File.ECG/1000;                % Señal ECG en mV
fs = File.fs;                       % Frecuencia de muestreo
clear File
N = length(ECG);                    % Número de muestras del ECG

% Simulamos una fuerte deriva de la línea de base 
t_w = (0:1/fs:(N-1)/fs)';
Deriva = 2.5*cos(2*pi*0.5*t_w);
ECG_Noise = ECG + Deriva;

% Definimos un filtro IIR Chebyshev-Type II de orden 8
orden = 6;                      % Orden del filtro
Rs = 40;                        % Atenuación en dB en la banda de rechazo
Wn = 2*0.6/fs;                  % Frecuencia de corte = 0.6 Hz
[b,a] = cheby2(orden,Rs,Wn,'high');

% Caracterizamos su respuesta en frecuencia H(f) y al impulso h[n]
f = 0:fs/N:fs-1/N;                      % Eje de frecuencias (Hz)
H = freqz(b,a,f,fs);                    % Respuesta en frecuencia
h = impz(b,a);                          % Respuesta al impulso

clf(figure(1))
subplot(211)
plot(f,20*log10(abs(H))),grid on
axis([0,3,-60,10])
xlabel('Frecuencia (Hz)','FontSize',FS)
ylabel('Magnitud (dB)','FontSize',FS)
title('Filtro IIR Chebyshev-Type II de orden 6 y fc = 0.6 Hz','FontSize',FS)
subplot(212)
plot(f,angle(H)),grid on
xlim([0,3])
xlabel('Frecuencia (Hz)','FontSize',FS)
ylabel('Fase (rad)','FontSize',FS)

% Filtrado directo e inverso con filtfilt
ECG_filtered = filtfilt(b,a,ECG_Noise);

% Comparación de la señal ECG sintetizada con la filtrada
clf(figure(2))
plot(t_w,ECG_Noise,t_w,ECG_filtered),grid on
lgnd = legend('Original','Filtrada IIR','filtfilt');
set(lgnd,'FontSize',FS-2,'Location','SouthEast')
set(gca,'xtick',0:5,'FontSize',FS-2)
xlabel('Tiempo (s)','FontSize',FS)
ylabel('Amplitud (mV)','FontSize',FS)
title('Señal ECG con deriva de base sintetizada','FontSize',FS)

%% Sección 3.2.2.3 Ajuste polinómico
clear,clc
FS = 12;

% Obtenemos un ECG de referencia
File = load('ECG_Example.mat');
ECG = File.ECG/1000;                % Señal ECG en mV
fs = File.fs;                       % Frecuencia de muestreo
clear File
N = length(ECG);                    % Número de muestras del ECG

% Simulamos una fuerte deriva de la línea de base 
t_w = (0:1/fs:(N-1)/fs)';
Deriva = 2.5*cos(2*pi*0.5*t_w);
ECG_Noise = ECG + Deriva;

% Identificamos puntos fiduciarios (primera columna en segundos)
M = [0.0804    2.0794;
     0.5318   -0.5935;
     1.0038   -2.6028;
     1.5339    0.2944;
     2.0161    2.5841;
     2.5633   -0.7243;
     3.0968   -2.6869;
     3.6850    0.9953;
     4.2254    1.5374;
     4.7657   -1.8925];

% Obtenemos el ajuste polinómico mediante spline cúbico
Ajuste = spline(M(:,1),M(:,2),t_w);

% Suprimimos el ajuste a la señal original
ECG_Denoised = ECG_Noise - Ajuste;

clf(figure(1))
subplot(211),hold on,grid on,box on
plot(t_w,ECG_Noise)
plot(M(:,1),M(:,2),'.k','MarkerSize',20)
plot(t_w,Ajuste,'r','LineWidth',1.5)
xlabel('Tiempo (s)','FontSize',FS)
ylabel('Amplitud (mv)','FontSize',FS)
title('Señal original y ajuste a puntos fiduciarios','FontSize',FS)
subplot(212)
plot(t_w,ECG_Denoised),grid on,box on
xlabel('Tiempo (s)','FontSize',FS)
ylabel('Amplitud (mv)','FontSize',FS)
title('Señal tras el ajuste','FontSize',FS)


%% Sección 3.2.3. Filtros IIR rechazo-banda para interferencia de red
clear,clc
FS = 12;

% Obtenemos un ECG de referencia
File = load('ECG_Example.mat');
ECG = File.ECG/1000;                % Señal ECG en mV
fs = File.fs;                       % Frecuencia de muestreo
clear File
N = length(ECG);                    % Número de muestras del ECG

% Simulamos una fuerte componente de 50 Hz 
t_w = (0:1/fs:(N-1)/fs)';
Noise = 0.5*cos(2*pi*50*t_w);
ECG_Noise = ECG + Noise;

% Coeficientes del filtro (se obtienen de diseñar los filtros a partir del 
% diagrama de ceros y polos en filterDesigner)
r = 0.95;
b = [ 0.996460094539525109702537974953884258866;
     -1.612328008172635662731408956460654735565;
      0.996460094539525109702537974953884258866];
switch r
    case 0.75
        a = [ 1;
             -1.213541829478161027111582370707765221596;
              0.5625                                   ];
    case 0.85
        a = [ 1;
             -1.375347406741916023165117621829267591238;
              0.722500000000000031086244689504383131862];
    case 0.95
        a = [ 1;
             -1.537152984005670797174047947919461876154;
              0.902500000000000079936057773011270910501];
    case 0.99
        a = [ 1;
             -1.601875214911172795595462048368062824011;
              0.980100000000000082245321664231596514583];
end

% Filtramos el ECG con el filtro seleccionado
ECG_Denoised = filter(b,a,ECG_Noise);

% Representamos el resultado
clf(figure(1)),grid on,box on, hold on
plot(t_w,ECG_Noise)
plot(t_w,ECG,'LineWidth',2)
xlabel('Tiempo (s)','FontSize',FS)
ylabel('Amplitud (mV)','FontSize',FS)
title('Señal sintetizada vs original')

figure(2),grid on,box on, hold on
subplot(133)
plot(t_w,ECG_Denoised)
xlim([0 1.5])
xlabel('Tiempo (s)','FontSize',FS)
ylabel('Amplitud (mV)','FontSize',FS)
title('r = 0.99')

%% Sección 3.2.4. Empirical Mode Decomposition (EMD)
% Requiere el 'Wavelet Toolbox'
clear,clc
FS = 12;

% Obtenemos un ECG de referencia
File = load('ECG_Example.mat');
ECG = File.ECG/1000;                % Señal ECG en mV
fs = File.fs;                       % Frecuencia de muestreo
clear File
N = length(ECG);                    % Número de muestras del ECG

% Simulamos una fuerte deriva de la línea de base 
t_w = (0:1/fs:(N-1)/fs)';
Deriva = 1*cos(2*pi*0.5*t_w);

% Simulamos una fuerte componente de 50 Hz
t_w = (0:1/fs:(N-1)/fs)';
Red = 0.5*cos(2*pi*50*t_w);

% Simulamos ruido miogénico en la banda [0-70] Hz
EMG = 1.0*randn(size(ECG));         % Ruido blanco
[b,a] = butter(6,2*[30,50]/fs);     % Filtro paso bajo fc = 70 Hz
EMG = filter(b,a,EMG);              % Ruido filtrado 
EMG = EMG.*cos(2*pi*0.2*t_w+pi/3);    % Ruido filtrado modulado en amplitud

% Contaminamos la señal ECG original con las tres fuentes de ruido
ECG_Noise = ECG + Deriva + Red + EMG;

% Método Empirical Mode Decomposition (EMD)
ConvergenceCriterion = 0.2;         % Umbral para detener iteraciones
Max_IMFs = 10;                      % Máximo de Intrinsic Mode Functions
Interpolation = 'spline';           % Método de interpolación (suave)
N_Extrema = 1;                      % Máximo de máximos y mínimos locales
Mostrar = 1;                        % Muestra la tabla con información

% Visualizamos el proceso de contaminación
sep = 3;    sep2 = 0.5;
clf(figure(2)),hold on,box on,grid on
plot(t_w,ECG);                text(0.25,sep2,'ECG original')
plot(t_w,Deriva-sep);         text(0.25,-sep+sep2,'Deriva línea base')
plot(t_w,Red-2*sep);          text(0.25,-2*sep+sep2,'Red')
plot(t_w,EMG-3*sep);          text(0.25,-3*sep+sep2,'EMG')
plot(t_w,ECG_Noise-4*sep);    text(0.25,-4*sep+sep2,'ECG ruidoso')
xlabel('Tiempo (s)')

[IMF,Residual,Info] = emd(ECG_Noise,'SiftRelativeTolerance',ConvergenceCriterion,...
                              'MaxNumIMF',Max_IMFs,...
                              'MaxNumExtrema',N_Extrema,...
                              'Interpolation',Interpolation,...
                              'Display',Mostrar);

% Visualise IMFs
clf(figure(1))
N_IMFs = size(IMF,2);
for c=1:N_IMFs+2
    subplot(N_IMFs+2,1,c),box on,grid on,
    if c==1,            plot(t_w,ECG_Noise); title('Señal original','FontSize',FS+2)
    elseif c==N_IMFs+2, plot(t_w,Residual); title('Residuo','FontSize',FS+2)
    else
        plot(t_w,IMF(:,c-1)); 
        title(sprintf('c_%d[n]  |  %d Iteraciones',c-1,...
            Info.NumSifting(c-1)),'FontSize',FS+2); 
    end
    if c==N_IMFs+2, xlabel('Tiempo (s)'); end
end

% Recomposición 
IMF_selected = 2:4;             % Para seleccionar todos: 1:N_IMFs
N_selected = length(IMF_selected);
ECG_Recomposed = zeros(size(ECG));
for c=1:N_selected
    ECG_Recomposed = ECG_Recomposed + IMF(:,IMF_selected(c));
end
% ECG_Recomposed = ECG_Recomposed + Residual;

% Visualización del resultado
clf(figure(3)),hold on,box on,grid on,
text(0.7,2,'Señal ECG original','FontSize',FS+2)
plot(t_w,ECG_Noise)
text(0.7,-3,'Señal ECG reconstruida con IMFs = [2,3,4]','FontSize',FS+2)
plot(t_w,ECG_Recomposed-4,'LineWidth',2)
xlabel('Tiempo (s)','FontSize',FS+2)

% Visualización del proceso para obtener c1[n]
[p_max,l_max] = findpeaks(ECG_Noise);
[p_min,l_min] = findpeaks(-ECG_Noise);
eu = spline((l_max-1)/fs,p_max,t_w);
el = spline((l_min-1)/fs,-p_min,t_w);
r1 = ECG_Noise - IMF(:,1);

clf(figure(4)),box on,grid on,hold on
text(0.1,2.5,'Señal ECG original x[n]','FontSize',FS)
plot(t_w,ECG_Noise)
plot((l_max-1)/fs,p_max,'.k','MarkerSize',10)
plot((l_min-1)/fs,-p_min,'.k','MarkerSize',10)
plot(t_w,eu)
plot(t_w,el)
plot(t_w,(eu+el)/2,'-k','LineWidth',2)
text(0.1,-1,'Señal IMF-1 c_1[n]','FontSize',FS)
plot(t_w,IMF(:,1)-2)
text(0.1,-5,'Residuo-1 r_1[n]','FontSize',FS)
plot(t_w,r1-5)
xlim([0 0.5])
xlabel('Tiempo (s)','FontSize',FS)

%% Sección 3.3.1. Algoritmo de Pan-Tompkins
clear,clc
FS = 12;

% Obtenemos un ECG de referencia
File = load('ECG_Example.mat');
ECG = File.ECG;                 % Señal ECG en mV
fs = File.fs;                   % Frecuencia de muestreo
clear File      
N = length(ECG);                % Número de muestras del ECG
t_w = (0:1/fs:(N-1)/fs)';       % Eje de tiempos

% Paso alto
orden = 4;                      % Orden del filtro
Rs = 40;                        % Atenuación en dB en la banda de rechazo
Wn = 2*0.4/fs;                  % Frecuencia de corte = 0.6 Hz
[b1,a1] = cheby2(orden,Rs,Wn,'high');

% Paso bajo
orden = 6;                      % Orden del filtro
Rs = 40;                        % Atenuación en dB en la banda de rechazo
Wn = 2*50/fs;                   % Frecuencia de corte = 50 Hz
[b2,a2] = cheby2(orden,Rs,Wn,'low');

% Función de transferencia de los filtros
% clf(figure(2)),freqz(b1,a1)
% clf(figure(2)),freqz(b2,a2)

% Filtrado
ECG_filt = filtfilt(b1,a1,ECG);             % Filtrado paso alto
ECG_filt = filtfilt(b2,a2,ECG_filt);        % Filtrado paso bajo

% Derivada (destaca la pendiente QRS)
ECG_deriv = diff(ECG_filt);                 % Derivada
ECG_deriv(end+1) = ECG_deriv(end);          % Forzamos mismo tamaño

% Elevamos al cuadrado
ECG_sq = ECG_deriv.^2;

% Integración con una ventena móvil (100 ms aprox)
L = round(0.100*fs);                    % Tamaño ventana (muestras)
h_int = (1/L)*ones(1,L);                % Ventana uniforme para promediar
ECG_int = conv(ECG_sq,h_int,'same');    % Integración

% Umbral y detección de picos
Threshold = 0.5*max(ECG_int);
[~,R] = findpeaks(ECG_int,'MinPeakHeight',Threshold,...
    'MinPeakDistance',round(0.250*fs));     % Max 240 lpm (60/0.250)

% Señal de posiciones
Positions = zeros(size(t_w));
Positions(R) = 1;

% Representación del resultado
sep = 1300;
clf(figure(1))
plot(t_w,ECG,'LineWidth',2),grid on,box on,hold on
plot(t_w,ECG_deriv*3-sep,'LineWidth',1)
plot(t_w,ECG_sq/20-2*sep,'LineWidth',1)
plot(t_w,ECG_int/5-3*sep,'LineWidth',1)
plot(t_w,Threshold/5-3*sep*ones(size(t_w)),'--k','LineWidth',1)
plot(t_w,Positions*500-4*sep,'LineWidth',1)
text(0.55,1400,'ECG original','FontSize',FS)
text(0.55,-sep+550,'Derivada (*3)','FontSize',FS)
text(0.55,-2*sep+800,'Derivada^2 (/20)','FontSize',FS)
text(0.55,-3*sep+900,'Integración (/5)','FontSize',FS)
text(0.55,-4*sep+800,'Detección','FontSize',FS)
xlabel('Tiempo (s)','FontSize',FS)
set(gca,'ytick',-5000:1000:2000,'yticklabel',{'','','','','','','',''})
xlim([0 3])
title('Algoritmo de Pan-Tompkins','FontSize',FS)


%% Sección 3.3.2.1 Transformada Wavelet Continua (Mexican-hat wavelet)
clear,clc

% Mexican-hat wavelet
fs = 500;
t_w = (-1:1/fs:1-1/fs)*1e2;
psi = (1-t_w.^2).*exp(-t_w.^2/2);
clf(figure(1))
plot(t_w,psi),grid on
xlabel('Tiempo (s)','FontSize',12)
title('Mexican-hat wavelet \psi(t)','FontSize',12)

% Obtenemos un ECG de referencia
File = load('ECG_Example.mat');
ECG = File.ECG;                 % Señal ECG en mV
fs = File.fs;                   % Frecuencia de muestreo
clear File      
N = length(ECG);                % Número de muestras del ECG
t = (0:1/fs:(N-1)/fs)';         % Eje de tiempos

% Señales filtradas
a=2^0;    psi_a0 = (1-(t_w/a).^2).*exp(-(t_w/a).^2/2);
y_a0 = 1/sqrt(a)*conv(ECG,psi_a0,'same');
a=2^2;    psi_a2 = (1-(t_w/a).^2).*exp(-(t_w/a).^2/2);
y_a2 = 1/sqrt(a)*conv(ECG,psi_a2,'same');
a=2^9;    psi_a9 = (1-(t_w/a).^2).*exp(-(t_w/a).^2/2);
y_a9 = 1/sqrt(a)*conv(ECG,psi_a9,'same');

% Representación del resultado
clf(figure(2))
subplot(221), plot(t,ECG)
xlabel('Tiempo (s)','FontSize',12)
title('ECG original','FontSize',12)
subplot(222), plot(t,y_a0)
xlabel('Tiempo (s)','FontSize',12)
title('a = 2^0','FontSize',12)
subplot(223), plot(t,y_a2)
xlabel('Tiempo (s)','FontSize',12)
title('a = 2^2','FontSize',12)
subplot(224), plot(t,y_a9)
xlabel('Tiempo (s)','FontSize',12)
title('a = 2^9','FontSize',12)


%% Sección 3.3.2.2. Transformada Wavelet Discreta (Daubechies-4 wavelet)
% Requiere el 'Wavelet Toolbox'
clear,clc
FS = 20;
WaveletMadre = 'db4';           % Daubechies-4 (db4)
Niveles = 8;                    % Niveles de descomposición

% Obtenemos un ECG de referencia
File = load('ECG_Example.mat');
ECG = File.ECG/1e3;             % Señal ECG en mV
fs = File.fs;                   % Frecuencia de muestreo
clear File      
N = length(ECG);                % Número de muestras del ECG
t = (0:1/fs:(N-1)/fs)';         % Eje de tiempos

% Simulamos una fuerte deriva de la línea de base 
Deriva = 0.5*cos(2*pi*0.5*t);

% Simulamos una fuerte componente de 50 Hz
Red = 0.5*cos(2*pi*50*t);

% Simulamos ruido miogénico en la banda [20-50] Hz
EMG = 0.5*randn(size(ECG));         % Ruido blanco
[b,a] = butter(6,2*[20,50]/fs);     % Filtro paso bajo fc = 70 Hz
EMG = filter(b,a,EMG);              % Ruido filtrado 
EMG = EMG.*cos(2*pi*0.2*t+pi/3);    % Ruido filtrado modulado en amplitud

% Contaminamos la señal ECG original con las tres fuentes de ruido
ECG = ECG + Deriva + Red + EMG;

% Descomponemos el ECG en los coefs de aproximación (a) y de detalle (d)
[c,l] = wavedec(ECG,Niveles,WaveletMadre);      % Wavelet decomposition
l_sum = cumsum(l);                      % Suma acumulada de posiciones
[d] = detcoef(c,l,1:Niveles,'cell');    % Coefs de detalle
a = c(1:l(1));                          % Coef de aproximación

% Seleccionamos y suprimimos coeficientes
SuprimirCoefs = 1;
if SuprimirCoefs
    Coefs2Supr = [1:3,7:Niveles];       %#ok
    for k=1:length(Coefs2Supr)
        d{Coefs2Supr(k)} = zeros(size(d{Coefs2Supr(k)}));   % Coefs de detalle
    end
    a = zeros(size(a));                             % Coef de aproximación
end

% Reconstruimos el vector de coeficientes (c)
c = zeros(size(c));
c(1:l(1)) = a;
for k=1:Niveles
    c(l_sum(k)+1:l_sum(k+1)) = d{Niveles-k+1};
end

% Reconstrucción del ECG mediante DWT
ECG_r = waverec(c,l,'db4');

% Detección de picos mediante umbral adaptativo
Threshold = 0.4*max(ECG_r);
[~,R] = findpeaks(ECG_r,'MinPeakHeight',Threshold,...
    'MinPeakDistance',round(0.250*fs));     % Max 240 ppm (60/0.250)

% Señal de posiciones
Positions = zeros(size(ECG));
Positions(R) = 1;

% Representación de coeficientes
clf(figure(1))
tiledlayout(Niveles+4,1,'Padding','compact','TileSpacing','compact')
nexttile,   plot(ECG),     ylabel('ECG','FontSize',FS) 
for k=1:Niveles
    nexttile
    plot(d{k})
    ylabel(sprintf('d%d',k),'FontSize',FS)
    xlim([1 l(Niveles+2-k)])
end
nexttile, plot(a),          ylabel('a','FontSize',FS), xlim([1 l(1)])
nexttile, plot(ECG_r),      ylabel('Reconstrucción','FontSize',FS)
nexttile, plot(Positions),  ylabel('Detección','FontSize',FS)
xlabel('Muestras','FontSize',FS)

% Representamos la wavelet madre Daubechies-4 (db4)
if 0
    Wavelet = 'db4';
    switch Wavelet
        case 'db4', titulo = 'Daubechies 4 (db4)';
        case 'sym4', titulo = 'Symlet 4 (sym4)';
    end
    [phi,psi,xval] = wavefun(Wavelet,10);
        % phi: función de escala
        % psi: wavelet madre
        % xval: eje del tiempo
        % 10 es la resolución de la wavelet (cuanto mayor, más precisa)
    clf(figure(2))
    plot(xval,psi,'LineWidth',2),grid on
    xlabel('Tiempo (s)','FontSize',FS)
    ylabel('Wavelet madre \psi(t)','FontSize',FS)
    title(titulo,'FontSize',FS)
end

%% Sección 3.3.3. Promediación con compensación de retardo
% Ejecutar Sección 3.3.1 (Pan-Tompkins) para encontrar los picos R

N_R = length(R);                        % Número de latidos detectados
PQRST_L = round([0.2,0.3]*fs);          % Tamaño (muestras) pre- y post-R
t_PQRST = (-PQRST_L(1):(PQRST_L(2)-1))/fs*1e3;      % Eje de t (ms)
PQRST = zeros(1,sum(PQRST_L));          % Inicialización
if N_R>3                                % Al menos 3 latidos son necesarios
    Template = ECG_filt(R(2)-PQRST_L(1)+1:R(2)+PQRST_L(2));  % Latido 2
    for k=2:N_R-1               % No consideramos el primer y último latido
        PQRST_local = ECG_filt(R(k)-PQRST_L(1)+1:R(k)+PQRST_L(2));
        [r,Delay] = xcorr(PQRST_local,Template);    % Correlación cruzada
        [~,pos] = max(r);      % Posición en el que se maximiza la correlación
        Delay = Delay(pos);    % Retraso en muestras a compensar

        % Visualizamos que es posible que exista retardo
        clf(figure(1))
        plot(t_PQRST,Template,t_PQRST,PQRST_local),grid on
        legend('Señal de referencia','Señal a desplazar')
        title(sprintf('ANTES de compensar   |  Conjunto %d/%d  |  Retraso en muestras = %d',k-1,N_R-2,Delay))

        % Compensamos el retardo y sumamos los segmentos de ECG
        PQRST_local = ECG_filt(R(k)-PQRST_L(1)+1+Delay:R(k)+PQRST_L(2)+Delay);
        PQRST = PQRST + PQRST_local;
        
        % Volvemos a estimar el retardo (al estar compensado, comprobamos que es 0)
        [r,Delay] = xcorr(PQRST_local,Template);
        [~,pos] = max(r);
        Delay = Delay(pos);
        clf(figure(2))
        plot(t_PQRST,Template,t_PQRST,PQRST_local),grid on
        legend('Señal de referencia','Señal a desplazar')
        title(sprintf('DESPUÉS de compensar  |  Conjunto %d/%d  |  Retraso en muestras = %d',k-1,N_R-2,Delay))
        fprintf('Pulsa una tecla para ver el siguiente complejo P-QRS-T.\n')
        pause
    end
    PQRST = PQRST/(N_R-2);
end
clf(figure(3))
plot(t_PQRST,PQRST,'LineWidth',2),grid on,box on,hold on
xlabel('Tiempo (ms)','FontSize',FS)
ylabel('Amplitud (\muV)','FontSize',FS)
title(sprintf('Conjunto P-QRS-T promediado (%d promedios)',N_R-2),'FontSize',FS)
